//
//  AHKPerson.m
//  AH-K3001V Utilities
//
//  Created by Fujidana on Tue Feb 15 2005.
//  Copyright (c) 2005 Fujidana Kobo. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#import "AHKPerson.h"
#import "AHKMultiValue.h"
#import "NSString_FJNFullwidthHalfwidth.h"

const int kAHKPersonMaximumNumber = 500;
const int kAHKPersonMaximumGroupNumber = 20;
const int kAHKPersonMaximumNameLength = 24;
const int kAHKPersonMaximumPhoneticNameLength = 24;
const int kAHKPersonMaximumAddressValueLength = 40;
const int kAHKPersonMaximumUrlLength = 255;
const int kAHKPersonMaximumHobbyLength = 10;
const int kAHKPersonMaximumNoteLength = 80;

//@interface AHKPerson (Private)
//+ (AHKAddressType)addressLabelWithString:(NSString *)string;
//+ (AHKBloodType)bloodTypeWithString:(NSString *)string;
//+ (AHKConstellation)zodiacWithString:(NSString *)string;
//@end
//
//#pragma mark -


static int deriving_signs_of_zodiac_option = AHKDerivingSignsOfZodiacNo;


@implementation AHKPerson

#pragma mark class methods

+ (void)initialize
{
	[self setKeys:[NSArray arrayWithObject:@"birthday"] triggerChangeNotificationsForDependentKey:@"zodiac"];
}

+ (int)derivingSignsOfZodiacOption
{
	return deriving_signs_of_zodiac_option;
}

+ (void)setDerivingSignsOfZodiacOption:(int)value
{
	deriving_signs_of_zodiac_option = value;
}

#pragma mark initialize methods

- (id)init
{
	self = [super init];
	if (self != nil) {
		[self setNumber:0];
		[self setGroupNumber:0];
		[self setPhones:[[[AHKPhoneNumbers alloc] init] autorelease]];
		[self setEmails:[[[AHKEmailAddresses alloc] init] autorelease]];
		[self setAddressLabel:AHKOtherAddressType];
		[self setZodiac:AHKUnclassifiedConstellation];
		[self setBloodType:AHKUnclassifiedBloodType];
	}
	return self;
}

- (void)dealloc
{
	[self setName:nil];
	[self setPhoneticName:nil];
	[self setPhones:nil];
	[self setEmails:nil];
	[self setAddressValue:nil];
	[self setUrl:nil];
	[self setBirthday:nil];
	[self setHobby:nil];
	[self setNote:nil];
	
	[super dealloc];
}

#pragma mark encode and decode methods

- (id)initWithCoder:(NSCoder *)coder
{
	self = [super init];
	if (self != nil) {
		[self setNumber:[coder decodeIntForKey:@"number"]];
		[self setGroupNumber:[coder decodeIntForKey:@"groupNumber"]];
		[self setName:[coder decodeObjectForKey:@"name"]];
		[self setPhoneticName:[coder decodeObjectForKey:@"phoneticName"]];
		[self setPhones:[coder decodeObjectForKey:@"phones"]];
		[self setEmails:[coder decodeObjectForKey:@"emails"]];
		[self setAddressLabel:[coder decodeIntForKey:@"addressLabelIndex"]];
		[self setAddressValue:[coder decodeObjectForKey:@"addressValue"]];
		[self setUrl:[coder decodeObjectForKey:@"url"]];
		[self setBloodType:[coder decodeIntForKey:@"bloodTypeIndex"]];
		[self setZodiac:[coder decodeIntForKey:@"constellationIndex"]];
		[self setBirthday:[coder decodeObjectForKey:@"birthday"]];
		[self setHobby:[coder decodeObjectForKey:@"hobby"]];
		[self setNote:[coder decodeObjectForKey:@"note"]];
		[self setSecured:[coder decodeBoolForKey:@"private"]];
	}
	return self;
}

- (void)encodeWithCoder:(NSCoder *)coder
{
	//	[super encodeWithCoder:coder];
	[coder encodeInt:number forKey:@"number"];
	[coder encodeInt:groupNumber forKey:@"groupNumber"];
	[coder encodeObject:name forKey:@"name"];
	[coder encodeObject:phoneticName forKey:@"phoneticName"];
	[coder encodeObject:phones forKey:@"phones"];
	[coder encodeObject:emails forKey:@"emails"];
	[coder encodeInt:addressLabel forKey:@"addressLabelIndex"];
	[coder encodeObject:addressValue forKey:@"addressValue"];
	[coder encodeObject:url forKey:@"url"];
	[coder encodeInt:bloodType forKey:@"bloodTypeIndex"];
	[coder encodeInt:zodiac forKey:@"constellationIndex"];
	[coder encodeObject:birthday forKey:@"birthday"];
	[coder encodeObject:hobby forKey:@"hobby"];
	[coder encodeObject:note forKey:@"note"];
	[coder encodeBool:secured forKey:@"private"];
}

#pragma mark accesser methods

- (int)number
{
	return number;
}

- (void)setNumber:(int)value
{
	number = value;
}

- (BOOL)validateNumber:(id *)ioValue error:(NSError **)outError
{
	if (*ioValue == nil) {
		*ioValue = [NSNumber numberWithInt:0];
		return YES;
	}
	if ([*ioValue intValue] < 0 || [*ioValue intValue] > kAHKPersonMaximumNumber - 1) {
		*ioValue = [NSNumber numberWithInt:0];
	}
	return YES;
}

- (int)groupNumber
{
	return groupNumber;
}

- (void)setGroupNumber:(int)value
{
	groupNumber = value;
}

- (BOOL)validateGroupNumber:(id *)ioValue error:(NSError **)outError
{
	if (*ioValue == nil) {
		*ioValue = [NSNumber numberWithInt:0];
		return YES;
	}
	if ([*ioValue intValue] < 0 || [*ioValue intValue] > kAHKPersonMaximumGroupNumber - 1) {
		*ioValue = [NSNumber numberWithInt:0];
		/*
		NSString *errorString = [NSString stringWithFormat:NSLocalizedStringFromTable(@"Number must be between 0 and %d.", @"AHKABLocalizable", @"validateGroupNumber"), kAHKPersonMaximumGroupNumber - 1];
		NSDictionary *userInfoDict = [NSDictionary dictionaryWithObject:errorString forKey:NSLocalizedDescriptionKey];
		NSError *error = [[[NSError alloc] initWithDomain:@"NUMBER_ERROR"	//NUMBER_ERROR_DOMAIN
													 code:0	//NUMBER_INVALID_CODE 
												 userInfo:userInfoDict] autorelease];
		*outError = error;
		return NO;
		*/
	}
	return YES;
}

- (NSString *)name
{
	return name;
}

- (void)setName:(NSString *)value
{
	if (name != value) {
		[name release];
		name = [value copy];
	}
}

- (BOOL)validateName:(id *)ioValue error:(NSError **)outError
{
	if (*ioValue == nil) {
		return YES;
	}
	if ([(NSString *)*ioValue length] > kAHKPersonMaximumNameLength) {
		*ioValue = [*ioValue substringToIndex:kAHKPersonMaximumNameLength];
	}
	return YES;
}

- (NSString *)phoneticName
{
	return phoneticName;
}

- (void)setPhoneticName:(NSString *)value
{
	if (phoneticName != value) {
		[phoneticName release];
		phoneticName = [value copy];
	}
}

- (BOOL)validatePhoneticName:(id *)ioValue error:(NSError **)outError
{
	if (*ioValue == nil) {
		return YES;
	}
	if ([*ioValue length] > kAHKPersonMaximumPhoneticNameLength) {
		*ioValue = [*ioValue substringToIndex:kAHKPersonMaximumPhoneticNameLength];
	}
	
	NSString *newString = [*ioValue halfwidthString];
	if ([newString isEqualToString:*ioValue] == NO) {
		*ioValue = newString;
	}
	return YES;
}

- (AHKPhoneNumbers *)phones
{
	return phones;
}

- (void)setPhones:(AHKPhoneNumbers *)value
{
	if (phones != value) {
		[phones release];
		phones = [value copy];
	}
}

- (AHKEmailAddresses *)emails
{
	return emails;
}

- (void)setEmails:(AHKEmailAddresses *)value
{
	if (emails != value) {
		[emails release];
		emails = [value copy];
	}
}

- (AHKAddressType)addressLabel
{
	return addressLabel;
}

- (void)setAddressLabel:(AHKAddressType)value
{
	addressLabel = value;
}

- (NSString *)addressValue
{
	return addressValue;
}

- (void)setAddressValue:(NSString *)value
{
	if (addressValue != value) {
		[addressValue release];
		addressValue = [value copy];
	}
}

- (BOOL)validateAddressValue:(id *)ioValue error:(NSError **)outError
{
	if (*ioValue == nil) {
		return YES;
	}
	if ([(NSString *)*ioValue length] > kAHKPersonMaximumAddressValueLength) {
		*ioValue = [*ioValue substringToIndex:kAHKPersonMaximumAddressValueLength];
	}
	return YES;
}

- (NSString *)url
{
	return url;
}

- (void)setUrl:(NSString *)value
{
	if (url != value) {
		[url release];
		url = [value copy];
	}
}

- (BOOL)validateUrl:(id *)ioValue error:(NSError **)outError
{
	if (*ioValue == nil) {
		return YES;
	}
	if ([(NSString *)*ioValue length] > kAHKPersonMaximumUrlLength) {
		*ioValue = [*ioValue substringToIndex:kAHKPersonMaximumUrlLength];
	}
	return YES;
}

- (AHKBloodType)bloodType
{
	return bloodType;
}

- (void)setBloodType:(AHKBloodType)value
{
	bloodType = value;
}

- (BOOL)validateBloodType:(id *)ioValue error:(NSError **)outError
{
	if (*ioValue == nil) {
		return YES;
	}
	if ([*ioValue intValue] < 0 || [*ioValue intValue] > 4) {
		*ioValue = [NSNumber numberWithInt:AHKUnclassifiedBloodType];
	}
	return YES;
}

- (AHKConstellation)zodiac
{
	return zodiac;
}

- (void)setZodiac:(AHKConstellation)value
{
	zodiac = value;
}

- (BOOL)validateZodiac:(id *)ioValue error:(NSError **)outError
{
	if (*ioValue == nil) {
		return YES;
	}
	if ([*ioValue intValue] < 0 || [*ioValue intValue] > 13) {
		*ioValue = [NSNumber numberWithInt:AHKUnclassifiedConstellation];
	}
	return YES;
}

// accessor for birthday
- (NSCalendarDate *)birthday
{
	return birthday;
}

- (void)setBirthday:(NSCalendarDate *)value
{
	if (birthday != value) {
		[birthday release];
		birthday = [value copy];
		
		if (birthday != nil) {
			int monthAndDay = [birthday monthOfYear] * 100 + [birthday dayOfMonth];
			
			// using 12 constellations
			if ([AHKPerson derivingSignsOfZodiacOption] == AHKDerivingSignsOfZodiacFrom12Constellation) {
				if (monthAndDay >= 1222 || monthAndDay <= 119) {
					[self setZodiac:AHKCapricornConstellation];
				} else if (monthAndDay <= 218) {
					[self setZodiac:AHKAquariusConstellation];
				} else if (monthAndDay <= 320) {
					[self setZodiac:AHKPiscesConstellation];
				} else if (monthAndDay <= 419) {
					[self setZodiac:AHKAriesConstellation];
				} else if (monthAndDay <= 520) {
					[self setZodiac:AHKTaurusConstellation];
				} else if (monthAndDay <= 621) {
					[self setZodiac:AHKGeminiConstellation];
				} else if (monthAndDay <= 722) {
					[self setZodiac:AHKCancerConstellation];
				} else if (monthAndDay <= 822) {
					[self setZodiac:AHKLeoConstellation];
				} else if (monthAndDay <= 922) {
					[self setZodiac:AHKVirgoConstellation];
				} else if (monthAndDay <= 1023) {
					[self setZodiac:AHKLibraConstellation];
				} else if (monthAndDay <= 1121) {
					[self setZodiac:AHKScorpioConstellation];
				} else if (monthAndDay <= 1221) {
					[self setZodiac:AHKSagittariusConstellation];
				}
			// using 13 constellations
			} else if ([AHKPerson derivingSignsOfZodiacOption] == AHKDerivingSignsOfZodiacFrom13Constellation) {
				if (monthAndDay >= 1218 || monthAndDay <= 119) {
					[self setZodiac:AHKSagittariusConstellation];
				} else if (monthAndDay <= 216) {
					[self setZodiac:AHKCapricornConstellation];
				} else if (monthAndDay <= 312) {
					[self setZodiac:AHKAquariusConstellation];
				} else if (monthAndDay <= 418) {
					[self setZodiac:AHKPiscesConstellation];
				} else if (monthAndDay <= 514) {
					[self setZodiac:AHKAriesConstellation];
				} else if (monthAndDay <= 621) {
					[self setZodiac:AHKTaurusConstellation];
				} else if (monthAndDay <= 720) {
					[self setZodiac:AHKGeminiConstellation];
				} else if (monthAndDay <= 810) {
					[self setZodiac:AHKCancerConstellation];
				} else if (monthAndDay <= 917) {
					[self setZodiac:AHKLeoConstellation];
				} else if (monthAndDay <= 1031) {
					[self setZodiac:AHKVirgoConstellation];
				} else if (monthAndDay <= 1123) {
					[self setZodiac:AHKLibraConstellation];
				} else if (monthAndDay <= 1129) {
					[self setZodiac:AHKScorpioConstellation];
				} else if (monthAndDay <= 1217) {
					[self setZodiac:AHKOphiuchusConstellation];
				}
			}
		}
	}
}

- (BOOL)validateBirthday:(id *)ioValue error:(NSError **)outError
{
	/*
	if (*ioValue == nil) {
		NSTimeInterval timeInterval = - [[NSTimeZone defaultTimeZone] secondsFromGMTForDate:*ioValue];
		*ioValue = [[[NSDate alloc] initWithTimeInterval:timeInterval
											   sinceDate:*ioValue] autorelease];
	}
	*/
	return YES;
}

// accessor for hobby
- (NSString *)hobby
{
	return hobby;
}

- (void)setHobby:(NSString *)value
{
	if (hobby != value) {
		[hobby release];
		hobby = [value copy];
	}
}

- (BOOL)validateHobby:(id *)ioValue error:(NSError **)outError
{
	if (*ioValue == nil) {
		return YES;
	}
	if ([(NSString *)*ioValue length] > kAHKPersonMaximumHobbyLength) {
		*ioValue = [*ioValue substringToIndex:kAHKPersonMaximumHobbyLength];
	}
	return YES;
}

// accessor for note
- (NSString *)note
{
	return note;
}

- (void)setNote:(NSString *)value
{
	if (note != value) {
		[note release];
		note = [value copy];
	}
}

- (BOOL)validateNote:(id *)ioValue error:(NSError **)outError
{
	if (*ioValue == nil) {
		return YES;
	}
	if ([(NSString *)*ioValue length] > kAHKPersonMaximumNoteLength) {
		*ioValue = [*ioValue substringToIndex:kAHKPersonMaximumNoteLength];
	}
	return YES;
}

- (BOOL)isSecured
{
	return secured;
}

- (void)setSecured:(BOOL)flag
{
	secured = flag;
}

#pragma mark other methods

- (NSString *)addressLabelString
{
	switch ([self addressLabel]) {
		case AHKHomeAddressType	: return @"HOME";
		case AHKWorkAddressType	: return @"WORK";
		default					: return @"POSTAL";
	}
}

- (void)setAddressLabelString:(NSString *)value
{
	NSString	*uppercaseString = [value uppercaseString];
	
	if ([uppercaseString isEqualToString:@"HOME"]) {
		[self setAddressLabel:AHKHomeAddressType];
	} else if ([uppercaseString isEqualToString:@"WORK"]) {
		[self setAddressLabel:AHKWorkAddressType];
	} else {
		[self setAddressLabel:AHKOtherAddressType];
	}
}

- (NSString *)bloodTypeString
{
	switch ([self bloodType]) {
		case AHKABloodType	: return @"A";
		case AHKBBloodType	: return @"B";
		case AHKABBloodType	: return @"AB";
		case AHKOBloodType	: return @"O";
		default				: return nil;
	}
}

- (void)setBloodTypeString:(NSString *)value
{
	NSString *uppercaseString = [value uppercaseString];
	
	if ([uppercaseString isEqualToString:@"A"]) {
		[self setBloodType:AHKABloodType];
	} else if ([uppercaseString isEqualToString:@"B"]) {
		[self setBloodType:AHKBBloodType];
	} else if ([uppercaseString isEqualToString:@"AB"]) {
		[self setBloodType:AHKABBloodType];
	} else if ([uppercaseString isEqualToString:@"O"]) {
		[self setBloodType:AHKOBloodType];
	} else {
		[self setBloodType:AHKUnclassifiedBloodType];
	}
}

- (NSString *)zodiacString
{
	switch ([self zodiac]) {
		case AHKSagittariusConstellation: return @"SAGITTARIUS";
		case AHKCapricornConstellation	: return @"CAPRICORN";
		case AHKAquariusConstellation	: return @"AQUARIUS";
		case AHKPiscesConstellation		: return @"PISCES";
		case AHKAriesConstellation		: return @"ARIES";
		case AHKTaurusConstellation		: return @"TAURUS";
		case AHKGeminiConstellation		: return @"GEMINI";
		case AHKCancerConstellation		: return @"CANCER";
		case AHKLeoConstellation		: return @"LEO";
		case AHKVirgoConstellation		: return @"VIRGO";
		case AHKLibraConstellation		: return @"LIBRA";
		case AHKScorpioConstellation	: return @"SCORPIO";
		case AHKOphiuchusConstellation	: return @"OPHIUCHUS";
		default							: return nil;
	}
}

- (void)setZodiacString:(NSString *)value
{
	NSString *uppercaseString = [value uppercaseString];
	if ([uppercaseString isEqualToString:@"SAGITTARIUS"]) {
		[self setZodiac:AHKSagittariusConstellation];
	} else if ([uppercaseString isEqualToString:@"CAPRICORN"]) {
		[self setZodiac:AHKCapricornConstellation];
	} else if ([uppercaseString isEqualToString:@"AQUARIUS"]) {
		[self setZodiac:AHKAquariusConstellation];
	} else if ([uppercaseString isEqualToString:@"PISCES"]) {
		[self setZodiac:AHKPiscesConstellation];
	} else if ([uppercaseString isEqualToString:@"ARIES"]) {
		[self setZodiac:AHKAriesConstellation];
	} else if ([uppercaseString isEqualToString:@"TAURUS"]) {
		[self setZodiac:AHKTaurusConstellation];
	} else if ([uppercaseString isEqualToString:@"GEMINI"]) {
		[self setZodiac:AHKGeminiConstellation];
	} else if ([uppercaseString isEqualToString:@"CANCER"]) {
		[self setZodiac:AHKCancerConstellation];
	} else if ([uppercaseString isEqualToString:@"LEO"]) {
		[self setZodiac:AHKLeoConstellation];
	} else if ([uppercaseString isEqualToString:@"VIRGO"]) {
		[self setZodiac:AHKVirgoConstellation];
	} else if ([uppercaseString isEqualToString:@"LIBRA"]) {
		[self setZodiac:AHKLibraConstellation];
	} else if ([uppercaseString isEqualToString:@"SCORPIO"]) {
		[self setZodiac:AHKScorpioConstellation];
	} else if ([uppercaseString isEqualToString:@"OPHIUCHUS"]) {
		[self setZodiac:AHKOphiuchusConstellation];
	} else {
		[self setZodiac:AHKUnclassifiedConstellation];
	}
}

- (NSString *)tabSeparatedValues
{
	NSArray	*array = [NSArray arrayWithObjects:[NSNumber numberWithInt:[self number]], 
		[NSNumber numberWithInt:[self groupNumber]], 
		([self name] ? [self name] : @""), 
		([self phoneticName] ? [[self phoneticName] fullwidthString] : @""), 
		([[self phones] firstValue] ? [[self phones] labelStringAtIndex:0] : @""), 
		([[self phones] firstValue] ? [[self phones] valueAtIndex:0] : @""), 
		([[self phones] secondValue] ? [[self phones] labelStringAtIndex:1] : @""), 
		([[self phones] secondValue] ? [[self phones] valueAtIndex:1] : @""), 
		([[self phones] thirdValue] ? [[self phones] labelStringAtIndex:1] : @""), 
		([[self phones] thirdValue] ? [[self phones] valueAtIndex:1] : @""), 
		[NSNumber numberWithUnsignedInt:[[self phones] indexForPreferredValue] + 1], 
		([[self emails] firstValue] ? [[self emails] labelStringAtIndex:0] : @""), 
		([[self emails] firstValue] ? [[self emails] valueAtIndex:0] : @""), 
		([[self emails] secondValue] ? [[self emails] labelStringAtIndex:1] : @""), 
		([[self emails] secondValue] ? [[self emails] valueAtIndex:1] : @""), 
		([[self emails] thirdValue] ? [[self emails] labelStringAtIndex:1] : @""), 
		([[self emails] thirdValue] ? [[self emails] valueAtIndex:1] : @""), 
		[NSNumber numberWithUnsignedInt:[[self emails] indexForPreferredValue] + 1], 
		([self addressValue] ? [self addressLabelString] : @""), 
		([self addressValue] ? [self addressValue] : @""), 
		([self url] ? [self url] : @""), 
		([self bloodTypeString] ?  [self bloodTypeString] : @""), 
		([self zodiacString] ? [self zodiacString] : @""), 
		([self birthday] ? [[self birthday] descriptionWithCalendarFormat:@"%y-%m-%d"] : @""), 
		([self hobby] ? [self hobby] : @""), 
		([self note] ? [self note] : @""), 
		([self isSecured] ? @"PRIVATE": @"PUBLIC"), 
		nil];
	return [array componentsJoinedByString:@"\t"];
}

+ (NSString *)tabSeparatedValuesHeader
{
	return (@"No.\t" 
		@"Gp\t"
		@"name\t" 
		@"phonetic name\t" 
		@"phone type 1\t" 
		@"phone value 1\t" 
		@"phone type 2\t" 
		@"phone value 2\t" 
		@"phone type 3\t" 
		@"phone value 3\t" 
		@"preferred phone index\t" 
		@"email type 1\t" 
		@"email value 1\t" 
		@"email type 2\t" 
		@"email value 2\t" 
		@"email type 3\t" 
		@"email value 3\t" 
		@"preferred email index\t" 
		@"address type\t" 
		@"address value\t" 
		@"url\t" 
		@"blood type\t" 
		@"sign of zodiac\t" 
		@"birthday\t" 
		@"hobby\t" 
		@"note\t" 
		@"class"); 
//	NSArray	*array = [NSArray arrayWithObjects:@"No.", 
//		@"Gp.", 
//		@"name", 
//		@"phonetic name", 
//		@"phone type 1", 
//		@"phone value 1", 
//		@"phone type 2", 
//		@"phone value 2", 
//		@"phone type 3", 
//		@"phone value 3", 
//		@"preferred phone index", 
//		@"email type 1", 
//		@"email value 1", 
//		@"email type 2", 
//		@"email value 2", 
//		@"email type 3", 
//		@"email value 3", 
//		@"preferred email index", 
//		@"address type", 
//		@"address value", 
//		@"url", 
//		@"blood type", 
//		@"sign of zodiac", 
//		@"birthday", 
//		@"hobby", 
//		@"note", 
//		@"class", 
//		nil];
//	return [array componentsJoinedByString:@"\t"];

}

@end
